use std::io::BufRead;


/// Implementation of a gamma compression function.  It’s takes a normalised
/// expanded value (a floating point in the range `[0.0, 1.0]`) and maps it to
/// compressed 8-bit integer.
trait Compress {
    /// Compresses normalised signal value into an 8-bit integer.
    ///
    /// Implementations of the trait must guarantee 1) that the function is
    /// non-decreasing (i.e. if `a > b` then `compress(a) >= compress(b)`) and
    /// 2) that for a `[0, 1]` domain its codomain is a full set of 8-bit
    /// unsigned integers (i.e. `∀ e ∈ {0..255} ∃ s ∈ [0, 1] f(s) = e`).
    fn compress(&self, value: f32) -> u8;
}

impl<C: Compress> Compress for &C {
    fn compress(&self, value: f32) -> u8 {
        (*self).compress(value)
    }
}


/// The normalised expanded value at which sRGB gamma function switches from
/// linear to power function.  Theoretically, it’s also a value at which linear
/// and power pieces of the gamma function give the same result.
const S_0: f32 = 0.003130668442500564;

/// Linear piece of sRGB gamma function.  In sRGB gamma it’s used in the `[0,
/// S_0]` domain.
fn linear_piece(s: f32) -> u8 {
    const D: f32 = 12.92 * 255.0;
    D.mul_add(s.max(0.0), 0.5) as u8
}

/// Power function piece of sRGB gamma function.  In sRGB gamma it’s used in the
/// `(S_0, 1.0]` domain.
fn power_piece(s: f32) -> u8 {
    const A: f32 = 0.055 * 255.0;
    const D: f32 = 1.055 * 255.0;
    D.mul_add(s.min(1.0).powf(5.0 / 12.0), -A + 0.5) as u8
}

impl Compress for () {
    /// Implementation of sRGB gamma compression function.
    fn compress(&self, value: f32) -> u8 {
        // Adding 0.5 is for rounding.
        if value <= S_0 {
            linear_piece(value)
        } else {
            power_piece(value)
        }
    }
}


/// For each 8-bit compressed value (except for 255) the largest normalised
/// value which maps to it.
struct Edges(pub [u32; 255]);

impl Edges {
    /// Constructs new object by calculating the edges using given function
    /// which maps normalised floats to compressed 8-bit integers.
    fn new(compress: impl Compress) -> Edges {
        let mut edges = [0; 255];
        let mut x = 0.0001;
        while compress.compress(x) != 0 {
            x *= 0.5;
            assert_ne!(x, 0.0);
        }
        let mut x = x.to_bits();
        edges[0] = x;
        loop {
            x += 1;
            let y = compress.compress(f32::from_bits(x));
            if y == 255 {
                break Self(edges);
            }
            edges[y as usize] = x;
        }
    }

    /// Returns the smallest value which maps to 255.
    fn min_255(&self) -> f32 { f32::from_bits(self.0[254] + 1) }

    /// Returns the largest difference between corresponding values in two
    /// `Edges` objects.
    fn compare(&self, other: &Edges) -> f32 {
        let mut max_diff = 0.0;
        for (a, b) in other.0.iter().zip(self.0.iter()) {
            let a = f32::from_bits(*a);
            let b = f32::from_bits(*b);
            let d = (a - b).abs();
            if d > max_diff {
                max_diff = d;
            }
        }
        max_diff
    }
}


/// Parameters for a LUT approximation.
struct ApproximationParams {
    /// Minimal value at which approximation formula will begin to be used.  For
    /// smaller values, the exact sRGB linear formula is used.
    min_approx: f32,
    /// Minimal value at which approximation starts returning constant 255.
    min_255: f32,
    /// Offset to subscribe from bits value of the floating number when
    /// calculating index in the LUT.
    bits_offset: u32,
    /// Steps in bits value of the floating number for consecutive values in
    /// LUT.
    bits_step: u32,
    /// Value added to Y values stored in LUT.
    y_offset: f32,
}

impl ApproximationParams {
    /// Verifies that parameters are valid.  Some checks are not strictly
    /// necessary and only verify that the values are in sensible range but some
    /// of the checks are required for LUT calculation and approximation to work
    /// correctly.
    fn verify(&self, min_255_limit: f32) -> Result<(), String> {
        let min_approx_limit = 0.001;
        if self.min_approx < min_approx_limit {
            return Err(format!(
                "min_approx ({}) too small; want >= {}",
                self.min_approx, min_approx_limit
            ));
        }
        if self.min_255 > min_255_limit {
            return Err(format!(
                "min_255 ({}) too large; want <= {}",
                self.min_255, min_255_limit));
        }
        if self.bits_offset > self.min_approx.to_bits() {
            return Err(format!(
                "bits_offset ({}) greater than min_approx ({})",
                f32::from_bits(self.bits_offset), self.min_approx));
        }
        if self.bits_step == 0 {
            return Err("bits_step may not be zero".to_owned());
        }
        if self.bits_offset + self.bits_step <= self.min_approx.to_bits() {
            return Err(
                format!("bits_offset ({:x}) too small / min_approx ({:x}) too large",
                        self.bits_offset, self.min_approx.to_bits()));
        }
        let entries = (self.min_255.to_bits() - self.bits_offset) / self.bits_step;
        if entries >= 500 {
            return Err(format!("too many entries (~{})", entries));
        }
        Ok(())
    }

    /// Calculates a LUT for the approximation based on the parameters.  May
    /// panic if parameters were not verified with [`verify`] method.  In
    /// particular, panics if `bits_offset` is greater than `min_approx`.
    fn calculate_lut(&self) -> Vec<f32> {
        assert!(self.bits_offset <= self.min_approx.to_bits());

        let start = self.bits_offset;
        let end = self.min_255.to_bits() + self.bits_step;
        (start..end)
            .step_by(self.bits_step as usize)
            .map(|bits| {
                let value = f32::from_bits(bits);
                if !(value > 0.0) {
                    0.0
                } else if value <= S_0 {
                    const D: f32 = 12.92 * 255.0;
                    D.mul_add(value, self.y_offset)
                } else {
                    const A: f32 = 0.055 * 255.0;
                    const D: f32 = 1.055 * 255.0;
                    D.mul_add(value.powf(5.0 / 12.0), self.y_offset - A)
                }
            })
            .collect()
    }

    /// Compresses a normalised into an 8-bit integer.  
    fn compress(
}


struct Approximation {
    min_approx: f32,
    min_255: f32,
    bits_offset: u32,
    bits_step: u32,
    lut: Vec,
}

impl Approximation {
    fn new(params: ApproximationParams,
           min_255_limit: f32) -> Result<Self, String> {
        params.verify(min_255_limit)?;

        assert!(params.bits_offset <= params.min_approx.to_bits());

        let start = params.bits_offset;
        let end = params.min_255.to_bits() + params.bits_step;
        let lut = (start..end)
            .step_by(params.bits_step as usize)
            .map(|bits| {
                let value = f32::from_bits(bits);
                if !(value > 0.0) {
                    0.0
                } else if value <= S_0 {
                    const D: f32 = 12.92 * 255.0;
                    D.mul_add(value, params.y_offset)
                } else {
                    const A: f32 = 0.055 * 255.0;
                    const D: f32 = 1.055 * 255.0;
                    D.mul_add(value.powf(5.0 / 12.0), params.y_offset - A)
                }
            })
            .enumerate()
            .map(|(idx, value)| {
                f32::encode(19, idx, value)
            })
            .collect();
        Ok(Self {
            min_255: params.min_255,
            min_approx: params.min_approx,
            bits_offset: params.bits_offset,
            bits_step: params.bits_step,
            lut: lut
        })
    }

    fn score(&self, edges: &Edges, verbose: bool) -> f32 {
        fn fmt(value: f32) -> String {
            if value == 0.0 {
                "0         ".to_owned()
            } else {
                let bits = value.to_bits();
                let exp = (bits >> 23) as i32 - 127;
                let val = f32::from_bits((bits & 0x007fffff) | (127 << 23));
                format!("{:4.2}*2^{:<3}", val, exp)
            }
        }

        if verbose {
            eprintln!("=== LUT {} start:{}/{:x} step:{:x} lut:{} ({}B) ===",
                      f32::type_name(), self.min_approx, self.bits_offset,
                      self.bits_step, self.lut.len(),
                      self.lut.len() * std::mem::size_of::());
        }

        let max_diff = Edges::new(self).compare_against(edges, |idx, a, b, d| {
            if verbose {
                let sign = if a < b { '<' } else if a > b { '>' } else { '=' };
                let ilog = (d.to_bits() >> 23) as i32 - 127;
                let bar = if ilog <= -15 {
                    String::from("")
                } else {
                    format!(" {}", "═══".repeat((ilog + 15) as usize))
                };
                eprintln!("{:3}  {}  {:9.7} {} {:9.7}  ({:>3}){}",
                          idx, fmt(d), a, sign, b, ilog, bar);
            }
        });

        if verbose {
            eprintln!("max_diff: {}", fmt(max_diff));
        }

        max_diff
    }
}

impl Compress for Approximation {
    fn compress(&self, value: f32) -> u8 {
        if !(value >= self.min_approx) {  // Also handle NaN.
            linear_piece(value)
        } else if value >= self.min_255 {
            255
        } else {
            let bits = value.to_bits() - self.bits_offset;
            let lft_x = (bits / self.bits_step) as usize;
            let rht_x = lft_x + 1;

            let lft = self.lut[lft_x].decode(19, lft_x);
            let rht = self.lut[rht_x].decode(19, rht_x);

            let lft_x = f32::from_bits(self.bits_offset + (lft_x as u32) * self.bits_step);
            let rht_x = f32::from_bits(self.bits_offset + (rht_x as u32) * self.bits_step);

            let dx = rht_x - lft_x;
            let ox = value - lft_x;

            (lft + (rht - lft) * ox / dx) as u8
        }
    }
}




// const START_BITS: u32 = 0x3b4f8978;
// //const DENSITY_SWITCH_INDEX: usize = 1000;
// const MIN_NON_LINEAR_BITS: u32 = 0x3b50dd8b;



// fn print_lut(name: &str, lut: &Vec, shift: usize, start: u32) {
//     println!("static {}: [{}; {}] = {{", name, f32::type_name(), lut.len());
//     for (idx, value) in lut.iter().enumerate() {
//         value.println(shift, idx);
//     }
//     println!("}};");
// }


/// Find smallest value for which sRGB linear piece does not match actual sRGB
/// value when encoding in 8-bits.
fn find_min_non_linear() -> f32 {
    let mut bits = S_0.to_bits();
    loop {
        bits += 1;
        let lin = linear_piece(f32::from_bits(bits));
        let exp = power_piece(f32::from_bits(bits));
        if lin != exp {
            break f32::from_bits(bits);
        }
    }
}


fn main() {
    let edges = Edges::new(());
    let min_255 = edges.min_255();
    assert_eq!(255, power_piece(min_255));
    assert_eq!(254, power_piece(f32::from_bits(min_255.to_bits() - 1)));

    eprintln!("min_approx = {}, min_255 = {}", find_min_non_linear(), min_255);

    fn parse_next<T: std::str::FromStr<Err = impl std::fmt::Display>>(word: Option<&str>) -> Result<T, String> {
        word
            .ok_or_else(|| "unexpected end of line".to_owned())
            .and_then(|word| {
                T::from_str(word).map_err(|err| format!("{}: {}", word, err))
            })
    }

    fn parse_line(line: &str) -> Result<ApproximationParams, String> {
        let mut words = line.split(char::is_whitespace);
        let min_approx: f32 = parse_next(words.next())?;
        let min_255: f32 = parse_next(words.next())?;
        let offset: f32 = parse_next(words.next())?;
        let bits_step: u32 = parse_next(words.next())?;
        let y_offset: f32 = parse_next(words.next())?;
        let params = ApproximationParams {
            min_approx: min_approx,
            min_255: min_255,
            bits_offset: offset.to_bits(),
            bits_step: bits_step,
            y_offset: y_offset,
        };
        if let Some(word) = words.next() {
            Err(format!("{}: unexpected token", word))
        } else {
            Ok(params)
        }
    }

    fn print_score(mut out: impl std::io::Write, lineno: usize, score: Result<(f32, usize), String>) {
        match score {
            Ok((max_diff, len)) => {
                writeln!(out, "{} {}", max_diff, len).unwrap();
            },
            Err(msg) => {
                writeln!(out, "inf inf").unwrap();
                eprintln!("stdin:{}: {}", lineno, msg);
            }
        };
        out.flush().unwrap();
    }

    for (no, result) in std::io::stdin().lock().lines().enumerate() {
        let line = match result {
            Ok(line) => line,
            Err(err) => panic!("stdin:{}: {}", no + 1, err)
        };
        let score = parse_line(line.trim())
            .and_then(|params| Approximation::new(params, min_255))
            .and_then(|approx| {
                let max_diff = approx.score(&edges, false);
                Ok((max_diff, approx.lut.len()))
            });
        print_score(std::io::stdout().lock(), no + 1, score);
    }

    // println!("const MIN_NON_LINEAR_BITS: u32 = 0x{:08x};", MIN_NON_LINEAR_BITS);
    // println!("const U8_APPROX_START_BITS: u32 = 0x{:08x};", START_BITS);
    // println!("const MIN_255_BITS: u32 = 0x{:08x};", edges.min_255_bits());
    // println!("const U8_APPROX_SHIFT: u32 = {};", shift);

    // for shift in [/*18, 19, 20*/ 19] {
    //     //let lut = get_lut::<u16>(&edges, shift, START_BITS);
    //     // print_lut("LINEAR_TO_U8_LUT", &lut, shift, START_BITS);
    //     //benchmark_lut(&edges, &lut, shift, START_BITS);

    //     let lut = get_lut::(&edges, shift, START_BITS);
    //     // print_lut("LINEAR_TO_U8_LUT", &lut, shift, START_BITS);
    //     benchmark_lut(&edges, &lut, shift, START_BITS);
    // }
}
