import React, { useEffect, useState, useCallback, useRef, useMemo } from 'react'
import { Icon, Overlay } from '@blueprintjs/core'
import { TransformWrapper, TransformComponent } from 'react-zoom-pan-pinch'
import debounce from 'debounce'

import { onDownload } from '../message/messageFunctions'
import { runtime } from '../../runtime'
import { isImage, isVideo, isAudio } from '../attachment/Attachment'
import { getLogger } from '../../../shared/logger'
import { gitHubIssuesUrl } from '../../../shared/constants'
import { useInitEffect } from '../helpers/hooks'
import { preventDefault } from '../../../shared/util'
import { jumpToMessage } from '../helpers/ChatMethods'
import { BackendRemote, onDCEvent, Type } from '../../backend-com'
import { selectedAccountId } from '../../ScreenController'
import useTranslationFunction from '../../hooks/useTranslationFunction'
import useDialog from '../../hooks/useDialog'
import useContextMenu from '../../hooks/useContextMenu'

import type { DialogProps } from '../../contexts/DialogContext'

const log = getLogger('renderer/fullscreen_media')

export enum NeighboringMediaMode {
  Chat,
  Global,
  Off,
}

type Props = {
  msg: Type.Message
  neighboringMedia: NeighboringMediaMode
}

export default function FullscreenMedia(props: Props & DialogProps) {
  const accountId = selectedAccountId()
  const tx = useTranslationFunction()
  const { openDialog } = useDialog()
  const { onClose } = props

  const [msg, setMsg] = useState(props.msg)
  const resetImageZoom = useRef<(() => void) | null>(
    null
  ) as React.MutableRefObject<(() => void) | null>
  const previousNextMessageId = useRef<[number | null, number | null]>([
    null,
    null,
  ])
  const [showPreviousNextMessageButtons, setShowPrevNextMsgBtns] = useState({
    previous: false,
    next: false,
  })

  const [mediaMessageList, setMediaMessageList] = useState<number[]>([])
  useEffect(() => {
    const update = () => {
      if (props.neighboringMedia === NeighboringMediaMode.Off) {
        setMediaMessageList([])
      } else {
        const { viewType, chatId } = props.msg
        // workaround to get gifs and images into the same media list
        let additionalViewType: Type.Viewtype | null = null
        if (props.msg.viewType === 'Image') {
          additionalViewType = 'Gif'
        } else if (props.msg.viewType === 'Gif') {
          additionalViewType = 'Image'
        }
        const scope =
          props.neighboringMedia === NeighboringMediaMode.Global ? null : chatId
        BackendRemote.rpc
          .getChatMedia(accountId, scope, viewType, additionalViewType, null)
          .then(setMediaMessageList)
      }
    }
    update()
    const debouncedUpdate = debounce(update, 400)
    const listeners = [
      onDCEvent(accountId, 'MsgsChanged', debouncedUpdate),
      onDCEvent(accountId, 'IncomingMsgBunch', debouncedUpdate),
      onDCEvent(accountId, 'MsgDeleted', debouncedUpdate),
    ]
    return () => {
      listeners.every(cleanup => cleanup())
    }
  }, [props.msg, props.neighboringMedia, accountId])

  const { file, fileMime } = msg

  if (!file) {
    log.error('file attribute not set in message', msg)
    throw new Error('file attribute not set')
  }

  const openMenu = useContextMenu([
    {
      label: tx('menu_copy_image_to_clipboard'),
      action: () => {
        runtime.writeClipboardImage(file)
      },
    },
    {
      label: tx('save_as'),
      action: onDownload.bind(null, msg),
    },
    {
      label: tx('show_in_chat'),
      action: () => {
        jumpToMessage(msg.id)
        onClose()
      },
    },
  ])

  let elm = null

  if (isImage(fileMime)) {
    elm = (
      <div className='image-container'>
        <TransformWrapper initialScale={1}>
          {utils => {
            resetImageZoom.current = () => {
              utils.resetTransform()
            }
            return (
              <TransformComponent>
                <div
                  className='image-context-menu-container'
                  onContextMenu={openMenu}
                >
                  <img src={runtime.transformBlobURL(file)} />
                </div>
              </TransformComponent>
            )
          }}
        </TransformWrapper>
      </div>
    )
  } else if (isAudio(fileMime)) {
    elm = <audio src={runtime.transformBlobURL(file)} controls />
  } else if (isVideo(fileMime)) {
    elm = <video src={runtime.transformBlobURL(file)} controls autoPlay />
  } else if (!fileMime) {
    // no file mime
    elm = (
      <div>
        <p>Error: Unknown mimeType for {runtime.transformBlobURL(file)}</p>
        <p>mimeType is "{fileMime}"</p>
        <p>
          Please report this bug on{' '}
          <a
            href='#'
            onClick={() => runtime.openLink(openDialog, gitHubIssuesUrl)}
          >
            github
          </a>
        </p>
      </div>
    )
    log.warn('Unknown mime type', { file, fileMime })
  } else {
    // can not be displayed by fullscreen media
    elm = (
      <div>
        <p>
          Error: Desktop issue: Unknown media type for{' '}
          {runtime.transformBlobURL(file)} (mimetype: {fileMime})
        </p>
        <p>
          Please report this bug on{' '}
          <a
            href='#'
            onClick={() => runtime.openLink(openDialog, gitHubIssuesUrl)}
          >
            github
          </a>
        </p>
      </div>
    )
    log.warn('Unknown media type for fullscreen media', {
      file,
      fileMime,
    })
  }
  const updatePreviousNextMessageId = useCallback(async () => {
    if (!msg.id) {
      return
    }
    previousNextMessageId.current = await getNeighboringMsgIds(
      msg.id,
      mediaMessageList
    )
    setShowPrevNextMsgBtns({
      previous: previousNextMessageId.current[0] !== null,
      next: previousNextMessageId.current[1] !== null,
    })
  }, [msg, mediaMessageList])

  useEffect(() => {
    updatePreviousNextMessageId()
  }, [msg, updatePreviousNextMessageId])
  useInitEffect(() => updatePreviousNextMessageId())

  const { previousImage, nextImage } = useMemo(() => {
    const loadMessage = async (msgID: number) => {
      if (msgID === 0) return
      const message = await BackendRemote.rpc.getMessage(
        selectedAccountId(),
        msgID
      )
      if (message === null) return
      setMsg(message)
      if (resetImageZoom.current !== null) {
        resetImageZoom.current()
      }
    }
    return {
      previousImage: () => loadMessage(previousNextMessageId.current[0] || 0),
      nextImage: () => loadMessage(previousNextMessageId.current[1] || 0),
    }
  }, [previousNextMessageId, setMsg])

  useEffect(() => {
    // use events directly for now
    // this will need adjustment to the keybindings manager once we have proper screen management
    // where we can know exactly which screen / menu / dialog is focused
    // and only send the context dependend keys to there
    const listener = (ev: KeyboardEvent) => {
      if (ev.repeat) {
        return
      }
      ev.preventDefault()
      ev.stopPropagation()
      if (ev.key == 'ArrowLeft') {
        previousImage()
      } else if (ev.key == 'ArrowRight') {
        nextImage()
      }
    }
    document.addEventListener('keydown', listener)
    return () => document.removeEventListener('keydown', listener)
  }, [previousImage, nextImage])

  if (!msg || !msg.file) return elm

  return (
    <Overlay
      isOpen={Boolean(file)}
      className='attachment-overlay'
      onClose={onClose}
    >
      <div className='render-media-wrapper' tabIndex={0}>
        <div className='attachment-view'>{elm}</div>
        {elm && (
          <div className='btn-wrapper'>
            <div
              role='button'
              onClick={onDownload.bind(null, msg)}
              className='download-btn'
              aria-label={tx('save')}
            />
            <Icon
              onClick={onClose}
              icon='cross'
              size={32}
              color={'grey'}
              aria-label={tx('close')}
            />
          </div>
        )}
        {showPreviousNextMessageButtons.previous && (
          <div className='media-previous-button'>
            <Icon
              onClick={preventDefault(previousImage)}
              icon='chevron-left'
              size={60}
            />
          </div>
        )}
        {showPreviousNextMessageButtons.next && (
          <div className='media-next-button'>
            <Icon
              onClick={preventDefault(nextImage)}
              icon='chevron-right'
              size={60}
            />
          </div>
        )}
      </div>
    </Overlay>
  )
}

async function getNeighboringMsgIds(messageId: number, list: number[]) {
  const index = list.indexOf(messageId)
  return [list[index - 1] || null, list[index + 1] || null] as [
    previousMessageId: number | null,
    nextMessageId: number | null,
  ]
}
