/*
 * Created by Angel Leon (@gubatron)
 * Copyright (c) 2011-2023, FrostWire(R). All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.limegroup.gnutella.gui;

import com.frostwire.desktop.DesktopPlatform;
import com.frostwire.jlibtorrent.LibTorrent;
import com.frostwire.platform.Platforms;
import com.frostwire.util.OSUtils;
import com.limegroup.gnutella.gui.bugs.FatalBugManager;
import com.limegroup.gnutella.util.FrostWireUtils;

import javax.swing.*;
import java.awt.*;
import java.io.*;
import java.util.Properties;

/**
 * This class constructs an <tt>Initializer</tt> instance that constructs
 * all of the necessary classes for the application.
 */
class GUILoader {
    /**
     * Creates an <tt>Initializer</tt> instance that constructs the
     * necessary classes for the application.
     * <p/>
     * <p>Invoked by com.limegroup.gnutella.gui.Main by reflection.
     *
     * @param args  the array of command line arguments
     * @param frame the splash screen; null, if no splash is displayed
     */
    public static void load(String[] args, Frame frame) {
        try {
            Platforms.set(new DesktopPlatform());
            System.out.printf("FrostWire version %s build %d\n", FrostWireUtils.getFrostWireVersion(), FrostWireUtils.getBuildNumber());
            System.out.println(System.getProperty("java.vm.name") + " " + System.getProperty("java.vendor") + " " + System.getProperty("java.version") + " " + System.getProperty("java.specification.vendor"));
            System.out.printf("JLibTorrent v%s\n", LibTorrent.jlibtorrentVersion());
            System.out.printf("Boost v%s\n", LibTorrent.boostVersion());
            System.out.printf("OpenSSL v%s\n", LibTorrent.opensslVersion());
            System.out.print("Windows App Store Installation: ");
            if (OSUtils.isWindowsAppStoreInstall()) {
                System.out.println("Yes");
            } else {
                System.out.println("Nope");
            }
            Initializer initializer = new Initializer();
            initializer.initialize(args, frame);
        } catch (Throwable err) {
            hideSplash(frame);
            try {
                FatalBugManager.handleFatalBug(err);
            } catch (Throwable t) {
                Throwable error = err;
                try {
                    t.initCause(err);
                    error = t;
                } catch (Throwable ignored) {
                }
                //System.out.println(t);       
                showCorruptionError(error);
                System.exit(1);
            }
        }
    }

    private static void hideSplash(Frame frame) {
        try {
            if (frame != null)
                frame.dispose();
            if (SplashWindow.isSplashConstructed())
                SplashWindow.instance().setVisible(false);
        } catch (Throwable ignored) {
        }
    }

    /**
     * Display a standardly formatted internal error message
     * coming from the backend.
     *
     * @param err the <tt>Throwable</tt> object containing information
     *            about the error
     */
    private static void showCorruptionError(Throwable err) {
        err.printStackTrace();
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        pw.println("FrostWire version " + FrostWireUtils.getFrostWireVersion());
        pw.print("Java version ");
        pw.print(System.getProperty("java.version", "?"));
        pw.print(" from ");
        pw.println(System.getProperty("java.vendor", "?"));
        pw.print(System.getProperty("os.name", "?"));
        pw.print(" v. ");
        pw.print(System.getProperty("os.version", "?"));
        pw.print(" on ");
        pw.println(System.getProperty("os.arch", "?"));
        Runtime runtime = Runtime.getRuntime();
        pw.println("Free/total memory: "
                + runtime.freeMemory() + "/" + runtime.totalMemory());
        pw.println();
        err.printStackTrace(pw);
        pw.println();
        pw.println("STARTUP ERROR!");
        pw.println();
        File propsFile = new File(getUserSettingsDir(), "frostwire.props");
        Properties props = new Properties();
        try {
            FileInputStream fis = new FileInputStream(propsFile);
            props.load(fis);
            fis.close();
            // list the properties in the PrintWriter.
            props.list(pw);
        } catch (IOException ignored) {
        }
        pw.flush();
        displayError(sw.toString());
    }

    /**
     * Gets the settings directory without using CommonUtils.
     */
    private static File getUserSettingsDir() {
        File dir = new File(System.getProperty("user.home"));
        String os = System.getProperty("os.name").toLowerCase();
        if (os.startsWith("mac os") && os.endsWith("x"))
            return new File(dir, "/Library/Preferences/FrostWire");
        else
            return new File(dir, ".frostwire");
    }

    /**
     * Displays an internal error with specialized formatting.
     */
    private static void displayError(String error) {
        System.out.println("Error: " + error);
        final JDialog DIALOG = new JDialog();
        DIALOG.setModal(true);
        final Dimension DIALOG_DIMENSION = new Dimension(350, 200);
        final Dimension INNER_SIZE = new Dimension(300, 150);
        DIALOG.setSize(DIALOG_DIMENSION);
        JPanel mainPanel = new JPanel();
        mainPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
        mainPanel.setLayout(new BorderLayout());
        String instr0;
        String instr1;
        String instr2;
        String instr3;
        String instr4;
        String instr5;
        instr0 = "One or more necessary files appear to be invalid.";
        instr1 = "This is generally caused by a corrupted installation.";
        instr2 = "Please try downloading and installing FrostWire again.";
        instr3 = "If the problem persists, please visit www.frostwire.com ";
        instr4 = "and click the 'Support' link.  ";
        instr5 = "Thank you.";
        JLabel label0 = new JLabel(instr0);
        JLabel label1 = new JLabel(instr1);
        JLabel label2 = new JLabel(instr2);
        JLabel label3 = new JLabel(instr3);
        JLabel label4 = new JLabel(instr4);
        JLabel label5 = new JLabel(instr5);
        JPanel labelPanel = new JPanel();
        JPanel innerLabelPanel = new JPanel();
        labelPanel.setLayout(new BoxLayout(labelPanel, BoxLayout.X_AXIS));
        innerLabelPanel.setLayout(new BoxLayout(innerLabelPanel, BoxLayout.Y_AXIS));
        innerLabelPanel.add(label0);
        innerLabelPanel.add(label1);
        innerLabelPanel.add(label2);
        innerLabelPanel.add(label3);
        innerLabelPanel.add(label4);
        innerLabelPanel.add(label5);
        innerLabelPanel.add(Box.createVerticalStrut(6));
        labelPanel.add(innerLabelPanel);
        labelPanel.add(Box.createHorizontalGlue());
        final JTextArea textArea = new JTextArea(error);
        textArea.selectAll();
        textArea.copy();
        textArea.setColumns(50);
        textArea.setEditable(false);
        JScrollPane scroller = new JScrollPane(textArea);
        scroller.setBorder(BorderFactory.createEtchedBorder());
        scroller.setPreferredSize(INNER_SIZE);
        JPanel buttonPanel = new JPanel();
        JButton copyButton = new JButton("Copy Report");
        copyButton.addActionListener(e -> {
            textArea.selectAll();
            textArea.copy();
        });
        JButton quitButton = new JButton("Ok");
        quitButton.addActionListener(e -> DIALOG.dispose());
        buttonPanel.add(copyButton);
        buttonPanel.add(quitButton);
        mainPanel.add(labelPanel, BorderLayout.NORTH);
        mainPanel.add(scroller, BorderLayout.CENTER);
        mainPanel.add(buttonPanel, BorderLayout.SOUTH);
        DIALOG.getContentPane().add(mainPanel);
        FatalBugManager.prepareDialog(DIALOG);
    }
}
