﻿/*
 * Copyright (C) 2022 Jaden Phil Nebel (Onionware)
 *
 * This file is part of OnionMedia.
 * OnionMedia is free software: you can redistribute it and/or modify it under the terms of the GNU Affero General Public License as published by the Free Software Foundation, version 3.

 * OnionMedia is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.

 * You should have received a copy of the GNU Affero General Public License along with OnionMedia. If not, see <https://www.gnu.org/licenses/>.
 */

using System;
using CommunityToolkit.Mvvm.ComponentModel;
using Microsoft.UI.Xaml.Navigation;
using OnionMedia.Contracts.Services;
using OnionMedia.Core;
using OnionMedia.Core.Models;
using OnionMedia.Core.Services;
using OnionMedia.Core.ViewModels;
using OnionMedia.Views;

namespace OnionMedia.ViewModels
{
    public class ShellViewModel : ObservableRecipient
    {
        private bool _isBackEnabled;
        private object _selected;

        public INavigationService NavigationService { get; }

        public INavigationViewService NavigationViewService { get; }

        public bool IsBackEnabled
        {
            get { return _isBackEnabled; }
            set { SetProperty(ref _isBackEnabled, value); }
        }

        public object Selected
        {
            get { return _selected; }
            set { SetProperty(ref _selected, value); }
        }

        public ShellViewModel(INavigationService navigationService, INavigationViewService navigationViewService)
        {
            NavigationService = navigationService;
            NavigationService.Navigated += OnNavigated;
            NavigationViewService = navigationViewService;
        }

        private void OnNavigated(object sender, NavigationEventArgs e)
        {
            IsBackEnabled = NavigationService.CanGoBack;
            AppSettings.Instance.DownloaderPageIsOpen = e.SourcePageType == typeof(YouTubeDownloaderPage);
            if (e.SourcePageType == typeof(SettingsPage))
            {
                Selected = NavigationViewService.SettingsItem;
                return;
            }

            var selectedItem = NavigationViewService.GetSelectedItem(e.SourcePageType);
            if (selectedItem == null) return;

            Selected = selectedItem;

            SetProgressBarSenderType(e.SourcePageType);
        }

        
        private void SetProgressBarSenderType(Type sourcePageType)
        {
            taskbarProgressService ??= IoC.Default.GetService<ITaskbarProgressService>();
            if (taskbarProgressService == null || sourcePageType == null) return;
            
            if (sourcePageType == typeof(MediaPage) && IoC.Default.GetService<MediaViewModel>()?.StartConversionCommand.IsRunning == true)
                taskbarProgressService.SetType(typeof(MediaViewModel));
            
            if (sourcePageType == typeof(YouTubeDownloaderPage) && IoC.Default.GetService<YouTubeDownloaderViewModel>()?.DownloadFileCommand.IsRunning == true)
                taskbarProgressService.SetType(typeof(YouTubeDownloaderViewModel));
        }
        private ITaskbarProgressService taskbarProgressService;
    }
}