// @ts-ignore[untyped-import]
import { x25519 } from "../internal/noble-curves-1.3.0.js"
import { random } from "../random/Randomizer.js"

export type EccPrivateKey = Uint8Array
export type EccPublicKey = Uint8Array

/**
 * Contains a public key and its corresponding private key
 *
 * NOTE: Keys should be cleared from memory once they are no longer needed!
 */
export type EccKeyPair = {
	publicKey: EccPublicKey
	privateKey: EccPrivateKey
}

/**
 * Contains all information for deriving AES keys
 *
 * The shared secret MUST NEVER be used as a key directly as it is a biased (some bits are more likely to be set than others).
 * The involved public keys should also be included when deriving an AES key from these shared secrets.
 */
export type EccSharedSecrets = {
	ephemeralSharedSecret: Uint8Array
	authSharedSecret: Uint8Array
}

// The number of bytes for a private key in the curve
// the byte length of the modulus
const X25519_N_BYTE_LENGTH = 32

/**
 * @return randomly generated X25519 key pair
 */
export function generateEccKeyPair(): EccKeyPair {
	// noble-curves appears to clamp the private key when using it, but not when generating it, so for safety,
	// we do not want to store it un-clamped in case we use a different implementation later
	const privateKey = clampPrivateKey(random.generateRandomData(X25519_N_BYTE_LENGTH))
	const publicKey = derivePublicKey(privateKey)
	return {
		privateKey,
		publicKey,
	}
}

/**
 * Derive a shared secret from the sender's private key and the recipient's public key to encrypt a message
 * @param senderIdentityPrivateKey	the sender's private identity key
 * @param ephemeralPrivateKey  the ephemeral private key generated by the sender for just one message (to one or more recipients)
 * @param recipientIdentityPublicKey the recipient's public identity key
 * @return the shared secrets
 */
export function eccEncapsulate(
	senderIdentityPrivateKey: EccPrivateKey,
	ephemeralPrivateKey: EccPrivateKey,
	recipientIdentityPublicKey: EccPublicKey,
): EccSharedSecrets {
	const ephemeralSharedSecret = generateSharedSecret(ephemeralPrivateKey, recipientIdentityPublicKey)
	const authSharedSecret = generateSharedSecret(senderIdentityPrivateKey, recipientIdentityPublicKey)
	return { ephemeralSharedSecret, authSharedSecret }
}

/**
 * Derive a shared secret from the recipient's private key and the sender's public key to decrypt a message
 * @param senderIdentityPublicKey	the sender's public identity key
 * @param ephemeralPublicKey  the ephemeral public key generated by the sender for just one message (to one or more recipients)
 * @param recipientIdentityPrivateKey the recipient's private identity key
 * @return shared secret and the sender's public key
 */
export function eccDecapsulate(
	senderIdentityPublicKey: EccPublicKey,
	ephemeralPublicKey: EccPublicKey,
	recipientIdentityPrivateKey: EccPrivateKey,
): EccSharedSecrets {
	const ephemeralSharedSecret = generateSharedSecret(recipientIdentityPrivateKey, ephemeralPublicKey)
	const authSharedSecret = generateSharedSecret(recipientIdentityPrivateKey, senderIdentityPublicKey)
	return { ephemeralSharedSecret, authSharedSecret }
}

/**
 * Diffie-Hellman key exchange; works by combining one party's private key and the other party's public key to form a shared secret between both parties
 */
function generateSharedSecret(localPrivateKey: EccPrivateKey, remotePublicKey: EccPublicKey): Uint8Array {
	const sharedSecret = x25519.getSharedSecret(localPrivateKey, remotePublicKey)

	// if every byte somehow happens to be 0, we can't use this as a secret; this is astronomically unlikely to happen by chance
	if (sharedSecret.every((val: Number) => val === 0)) {
		throw new Error("can't get shared secret: bad key inputs")
	}

	return sharedSecret
}

// see https://www.jcraige.com/an-explainer-on-ed25519-clamping for an explanation on why we do this
function clampPrivateKey(privateKey: EccPrivateKey): EccPrivateKey {
	// First, we want to unset the highest bit but set the second-highest bit to 1. This prevents potential timing and brute-force attacks, respectively.
	privateKey[privateKey.length - 1] = (privateKey[privateKey.length - 1] & 0b01111111) | 0b01000000

	// Then, we want to guarantee our scalar is a multiple of 8, our cofactor, to protect against small-subgroup attacks per RFC 2785 which could leak key data!
	privateKey[0] &= 0b11111000

	return privateKey
}

function derivePublicKey(privateKey: EccPrivateKey): EccPublicKey {
	return x25519.getPublicKey(privateKey)
}
