import o from "@tutao/otest"
import { decapsulate, encapsulate, generateKeyPair, LibOQSExports } from "../lib/encryption/Liboqs/Kyber.js"
import { random } from "../lib/index.js"
import { loadWasmModuleFallback, loadWasmModuleFromFile } from "./WebAssemblyTestUtils.js"
import { $ } from "zx"
import fs from "node:fs"

o.spec("Kyber", function () {
	// We need to generate the webassembly stuff during tests
	// since they're generated by esbuild during app build process
	// and packages tests run before the app build.
	o.before(async () => {
		const currentPath = process.cwd()

		const make = $
		make.verbose = false
		make.cwd = "../../../libs/webassembly/"
		make.env = {
			...process.env,
			WASM: `${currentPath}/liboqs.wasm`,
		}

		await make`make -f Makefile_liboqs build`
		await $`wasm2js ${currentPath}/liboqs.wasm -O3 --output ${currentPath}/liboqs.js`

		make.cwd = currentPath
	})

	// We remove them after the tests since we don't need them anymore
	o.after(() => {
		fs.rmSync("./liboqs.wasm")
		fs.rmSync("./liboqs.js")
	})

	o("encryption roundtrip", async function () {
		const liboqs = (await loadWasmModuleFromFile("./liboqs.wasm")) as LibOQSExports

		const keyPair = generateKeyPair(liboqs, random)
		o(keyPair.privateKey.raw.length).equals(3168)
		o(keyPair.publicKey.raw.length).equals(1568)

		const encapsulation = encapsulate(liboqs, keyPair.publicKey, random)
		o(encapsulation.sharedSecret.length).equals(32)
		o(encapsulation.ciphertext.length).equals(1568)

		const decapsulatedSecret = decapsulate(liboqs, keyPair.privateKey, encapsulation.ciphertext)

		o(encapsulation.sharedSecret).deepEquals(decapsulatedSecret)
	})

	o("encryption roundtrip - fallback", async function () {
		const liboqsFallback = (await loadWasmModuleFallback("../liboqs.js")) as LibOQSExports
		const keyPair = generateKeyPair(liboqsFallback, random)
		o(keyPair.privateKey.raw.length).equals(3168)
		o(keyPair.publicKey.raw.length).equals(1568)

		const encapsulation = encapsulate(liboqsFallback, keyPair.publicKey, random)
		o(encapsulation.sharedSecret.length).equals(32)
		o(encapsulation.ciphertext.length).equals(1568)

		const decapsulatedSecret = decapsulate(liboqsFallback, keyPair.privateKey, encapsulation.ciphertext)

		o(encapsulation.sharedSecret).deepEquals(decapsulatedSecret)
	})
})
