import m from "mithril"
import { Dialog, DialogType } from "../gui/base/Dialog"
import { lang } from "../misc/LanguageViewModel"
import { assertMainOrNode, isApp } from "../api/common/Env"
import { formatDate } from "../misc/Formatter"
import { HtmlEditor, HtmlEditorMode } from "../gui/editor/HtmlEditor"
import type { AccountingInfo, Customer, GroupInfo, OrderProcessingAgreement } from "../api/entities/sys/TypeRefs.js"
import { createSignOrderProcessingAgreementData } from "../api/entities/sys/TypeRefs.js"
import { neverNull } from "@tutao/tutanota-utils"
import { locator } from "../api/main/CommonLocator"
import { SignOrderProcessingAgreementService } from "../api/entities/sys/Services"
import { formatNameAndAddress } from "../api/common/utils/CommonFormatter.js"
import { getMailAddressDisplayText } from "../mailFunctionality/SharedMailUtils.js"

assertMainOrNode()
const PRINT_DIV_ID = "print-div"
const agreementTexts = {
	"1_en": {
		heading:
			'<div class="papertext"><h3 style="text-align: center;" id="Orderprocessingagreement-Orderprocessingagreement">Order processing agreement</h3><p style="text-align: center;">between</p>',
		content:
			'<p style="text-align: center;">-&nbsp;controller -<br>hereinafter referred to as the Client</p><p style="text-align: center;">and</p><p style="text-align: center;">Tutao GmbH, Deisterstr. 17a, 30449 Hannover, Germany</p><p style="text-align: center;">-&nbsp;processor -<br>hereinafter referred to as the Supplier</p><p style="text-align: center;">&nbsp;</p><h4 id="Orderprocessingagreement-1.Subjectmatteranddurationoftheagreement">1.&nbsp;Subject matter and duration of the agreement</h4><p>The Subject matter of the agreement results from the Terms and Conditions of Tutao GmbH in its current version, see <span class="nolink">https://tuta.com/terms</span>, which is referred to here (hereinafter referred to as Service Agreement). The Supplier processes personal data for the Client according to Art. 4 no. 2 and Art. 28 GDPR based on this agreement.</p><p>The duration of this Agreement corresponds to the selected term of policy in the selected tariff.</p><h4 id="Orderprocessingagreement-2.Purpose,TypeofDataandCategoriesofDataSubjects">2. Purpose, Type of Data and Categories of Data Subjects</h4><p>For the initiation of a contractual relationship and for service provision</p><ul><li>the newly registered email address</li></ul><p>is collected as inventory data.</p><p>For invoicing and determining the VAT</p><ul><li>the domicile of the customer (country)</li><li>the invoicing address</li><li>the VAT identification number (only for business customers of some countries)</li></ul><p>is collected as inventory data.</p><p>For the transaction of payments the following payment data (inventory data) is collected depending on the chosen payment method:</p><ul><li>Banking details (account number and sort code and IBAN/BIC, if necessary bank name, account holder),</li><li>credit card data,</li><li>PayPal user name.</li></ul><p>For the execution of direct debiting, the banking details are shared with the authorized credit institution. For the execution of PayPal payments, the PayPal data is shared with PayPal (Europe). For the execution of credit card payments, the credit card data is shared with the payment service provider&nbsp;Braintree&nbsp;for subprocessing. This includes the transfer of personal data into a third country (USA). An agreement entered into with Braintree defines appropriate safeguards and demands that the data is only processed in compliance with the GDPR and only for the purpose of execution of payments. This agreement can be examined here:&nbsp;<span class="nolink">https://www.braintreepayments.com/assets/Braintree-PSA-Model-Clauses-March2018.pdf</span></p><p>Tutanota provides services for saving, editing, presentation and electronic transmission of data, such as email service, contact management and data storage. Within the context of this content data, personal data of the Client may be processed. All textual content is encrypted for the user and its communication partners in a way that even Tutao GmbH has no access to the data.&nbsp;</p><p>In order to maintain email server operations, for error diagnosis and for prevention of abuse, mail server logs are stored max. 30 days. These logs contain sender and recipient email addresses and time of connection, but no customer IP addresses.&nbsp;</p><p>In order to maintain operations, for prevention of abuse and and for visitors analysis, IP addresses of users are processed. Storage only takes place for IP addresses made anonymous which are therefore not personal data any more.</p><p>With the exception of payment data, the personal data including the email address is not disclosed to third parties. However, Tutao GmbH can be legally bound to provide content data (in case of a valid German court order) and inventory data to prosecution services. There will be no sale of data.</p><p>The undertaking of the contractually agreed Processing of Data shall be carried out exclusively within a Member State of the European Union (EU) or within a Member State of the European Economic Area (EEA). Each and every Transfer of Data to a State which is not a Member State of either the EU or the EEA requires the prior agreement of the Client and shall only occur if the specific Conditions of Article 44 et seq. GDPR have been fulfilled.</p><p>The Categories of Data Subjects comprise the users set up in Tutanota by the Client and these users\' communication partners.</p><h4 id="Orderprocessingagreement-3.TechnicalandOrganizationalMeasures">3. Technical and Organizational Measures</h4><p>(1) Before the commencement of processing, the Supplier shall document the execution of the necessary Technical and Organizational Measures, set out in advance of the awarding of the Agreement, specifically with regard to the detailed execution of the Agreement, and shall present these documented measures to the Client for inspection. Upon acceptance by the Client, the documented measures become the foundation of the Agreement. Insofar as the inspection/audit by the Client shows the need for amendments, such amendments shall be implemented by mutual agreement.</p><p>(2) The Supplier shall establish the security in accordance with Article 28 Paragraph 3 Point c, and Article 32 GDPR in particular in conjunction with Article 5 Paragraph 1, and Paragraph 2 GDPR. The measures to be taken are measures of data security and measures that guarantee a protection level appropriate to the risk concerning confidentiality, integrity, availability and resilience of the systems. The state of the art, implementation costs, the nature, scope and purposes of processing as well as the probability of occurrence and the severity of the risk to the rights and freedoms of natural persons within the meaning of Article 32 Paragraph 1 GDPR must be taken into account. [Details in Appendix 1]</p><p>(3) The Technical and Organizational Measures are subject to technical progress and further development. In this respect, it is permissible for the Supplier to implement alternative adequate measures. In so doing, the security level of the defined measures must not be reduced. Substantial changes must be documented.</p><h4 id="Orderprocessingagreement-4.Rectification,restrictionanderasureofdata"><span>4. Rectification, restriction and erasure of data</span></h4><p>(1) The Supplier may not on its own authority rectify, erase or restrict the processing of data that is being processed on behalf of the Client, but only on documented instructions from the Client. <br>Insofar as a Data Subject contacts the Supplier directly concerning a rectification, erasure, or restriction of processing, the Supplier will immediately forward the Data Subject’s request to the Client.</p><p>(2) Insofar as it is included in the scope of services, the erasure policy, ‘right to be forgotten’, rectification, data portability and access shall be ensured by the Supplier in accordance with documented instructions from the Client without undue delay.</p><h4 id="Orderprocessingagreement-5.QualityassuranceandotherdutiesoftheSupplier">5. Quality assurance and other duties of the Supplier&nbsp;</h4><p align="justify">In addition to complying with the rules set out in this Agreement, the Supplier shall comply with the statutory requirements referred to in Articles 28 to 33 GDPR; accordingly, the Supplier ensures, in particular, compliance with the following requirements:</p><ol><li><p align="justify">The Supplier is not obliged to appoint a Data Protection Officer. Mr. Arne Moehle, phone: +49 511 202801-11, arne.moehle@tutao.de, is designated as the Contact Person on behalf of the Supplier.</p></li><li><p align="justify">Confidentiality in accordance with Article 28 Paragraph 3 Sentence 2 Point b, Articles 29 and 32 Paragraph 4 GDPR. The Supplier entrusts only such employees with the data processing outlined in this Agreement who have been bound to confidentiality and have previously been familiarized with the data protection provisions relevant to their work. The Supplier and any person acting under its authority who has access to personal data, shall not process that data unless on instructions from the Client, which includes the powers granted in this Agreement, unless required to do so by law.</p></li><li><p align="justify">Implementation of and compliance with all Technical and Organizational Measures necessary for this Agreement in accordance with Article 28 Paragraph 3 Sentence 2 Point c, Article 32 GDPR [details in Appendix 1].</p></li><li><p align="justify">The Client and the Supplier shall cooperate, on request, with the supervisory authority in performance of its tasks.</p></li><li><p align="justify">The Client shall be informed immediately of any inspections and measures conducted by the supervisory authority, insofar as they relate to this Agreement. This also applies insofar as the Supplier is under investigation or is party to an investigation by a competent authority in connection with infringements to any Civil or Criminal Law, or Administrative Rule or Regulation regarding the processing of personal data in connection with the processing of this Agreement.</p></li><li><p align="justify">Insofar as the Client is subject to an inspection by the supervisory authority, an administrative or summary offense or criminal procedure, a liability claim by a Data Subject or by a third party or any other claim in connection with the Agreement data processing by the Supplier, the Supplier shall make every effort to support the Client.</p></li><li><p align="justify">The Supplier shall periodically monitor the internal processes and the Technical and Organizational Measures to ensure that processing within his area of responsibility is in accordance with the requirements of applicable data protection law and the protection of the rights of the data subject.</p></li><li><p align="justify">Verifiability of the Technical and Organizational Measures conducted by the Client as part of the Client’s supervisory powers referred to in item 7 of this Agreement.</p></li></ol><h4 id="Orderprocessingagreement-6.Subcontracting">6. Subcontracting</h4><p align="justify">(1) Subcontracting for the purpose of this Agreement is to be understood as meaning services which relate directly to the provision of the principal service. This does not include ancillary services, such as telecommunication services, postal / transport services, maintenance and user support services or the disposal of data carriers, as well as other measures to ensure the confidentiality, availability, integrity and resilience of the hardware and software of data processing equipment. The Supplier shall, however, be obliged to make appropriate and legally binding contractual arrangements and take appropriate inspection measures to ensure the data protection and the data security of the Client\'s data, even in the case of outsourced ancillary services.</p><p align="justify">(2) The Supplier may commission subcontractors (additional contract processors) only after prior explicit written or documented consent from the Client.&nbsp;</p><p align="justify">(3) Outsourcing to subcontractors or changing the existing subcontractor are permissible when:</p><ul><li>The Supplier submits such an outsourcing to a subcontractor to the Client in writing or in text form with appropriate advance notice; and</li><li>The Client has not objected to the planned outsourcing in writing or in text form by the date of handing over the data to the Supplier; and</li><li>The subcontracting is based on a contractual agreement in accordance with Article 28 paragraphs 2-4 GDPR.</li></ul><p align="justify">(4) The transfer of personal data from the Client to the subcontractor and the subcontractors commencement of the data processing shall only be undertaken after compliance with all requirements has been achieved.</p><p align="justify">(5) If the subcontractor provides the agreed service outside the EU/EEA, the Supplier shall ensure compliance with EU Data Protection Regulations by appropriate measures. The same applies if service providers are to be used within the meaning of Paragraph 1 Sentence 2.</p><p align="justify">(6) Further outsourcing by the subcontractor requires the express consent of the main Client (at the minimum in text form);</p><p align="justify">(7) All contractual provisions in the contract chain shall be communicated to and agreed with each and every additional subcontractor.</p><h4 class="western" id="Orderprocessingagreement-7.SupervisorypowersoftheClient">7. Supervisory powers of the Client</h4><p align="justify">(1) The Client has the right, after consultation with the Supplier, to carry out inspections or to have them carried out by an auditor to be designated in each individual case. It has the right to convince itself of the compliance with this agreement by the Supplier in his business operations by means of random checks, which are ordinarily to be announced in good time.</p><p align="justify">(2) The Supplier shall ensure that the Client is able to verify compliance with the obligations of the Supplier in accordance with Article 28 GDPR. The Supplier undertakes to give the Client the necessary information on request and, in particular, to demonstrate the execution of the Technical and Organizational Measures.</p><p align="justify">(3) Evidence of such measures, which concern not only the specific Agreement, may be provided by</p><ul><li>Compliance with approved Codes of Conduct pursuant to Article 40 GDPR;</li><li>Certification according to an approved certification procedure in accordance with Article 42 GDPR;</li><li>Current auditor’s certificates, reports or excerpts from reports provided by independent bodies (e.g. auditor, Data Protection Officer, IT security department, data privacy auditor, quality auditor)</li><li>A suitable certification by IT security or data protection auditing (e.g. according to BSI-Grundschutz (IT Baseline Protection certification developed by the German&nbsp; Federal Office for Security in Information Technology (BSI)) or ISO/IEC 27001).</li></ul><p align="justify">(4) The Supplier may claim remuneration for enabling Client inspections.&nbsp;</p><h4 class="western" id="Orderprocessingagreement-8.CommunicationinthecaseofinfringementsbytheSupplier">8. Communication in the case of infringements by the Supplier</h4><p align="justify">(1) The Supplier shall assist the Client in complying with the obligations concerning the security of personal data, reporting requirements for data breaches, data protection impact assessments and prior consultations, referred to in Articles 32 to 36 of the GDPR. These include:</p><ol><li>Ensuring an appropriate level of protection through Technical and Organizational Measures that take into account the circumstances and purposes of the processing as well as the projected probability and severity of a possible infringement of the law as a result of security vulnerabilities and that enable an immediate detection of relevant infringement events.</li><li>The obligation to report a personal data breach immediately to the Client</li><li>The duty to assist the Client with regard to the Client’s obligation to provide information to the Data Subject concerned and to immediately provide the Client with all relevant information in this regard.</li><li>Supporting the Client with its data protection impact assessment</li><li>Supporting the Client with regard to prior consultation of the supervisory authority</li></ol><p align="justify">(2) The Supplier may claim compensation for support services which are not included in the description of the services and which are not attributable to failures on the part of the Supplier.</p><h4 class="western" id="Orderprocessingagreement-9.AuthorityoftheClienttoissueinstructions">9. Authority of the Client to issue instructions</h4><p>(1) The Client shall immediately confirm oral instructions (at the minimum in text form).</p><p>(2) The Supplier shall inform the Client immediately if he considers that an instruction violates Data Protection Regulations. The Supplier shall then be entitled to suspend the execution of the relevant instructions until the Client confirms or changes them.</p><h4 class="western" id="Orderprocessingagreement-10.Deletionandreturnofpersonaldata">10. Deletion and return of personal data</h4><p>(1) Copies or duplicates of the data shall never be created without the knowledge of the Client, with the exception of back-up copies as far as they are necessary to ensure orderly data processing, as well as data required to meet regulatory requirements to retain data.</p><p>(2) After conclusion of the contracted work, or earlier upon request by the Client, at the latest upon termination of the Service Agreement, the Supplier shall hand over to the Client or – subject to prior consent – destroy all documents, processing and utilization results, and data sets related to the Agreement that have come into its possession, in a data-protection compliant manner. The same applies to any and all connected test, waste, redundant and discarded material. The log of the destruction or deletion shall be provided on request.</p><p>(3) Documentation which is used to demonstrate orderly data processing in accordance with the Agreement shall be stored beyond the contract duration by the Supplier in accordance with the respective retention periods. It may hand such documentation over to the Client at the end of the contract duration to relieve the Supplier of this contractual obligation.</p><h4 id="Orderprocessingagreement-11.Finalprovisions">11. Final provisions</h4><p align="justify">(1) This agreement shall be governed by and construed in accordance with German law. Place of jurisdiction shall be Hanover, Germany.</p><p align="justify">(2) Any changes of or amendments to this Agreement must be in writing to become effective. This includes any alteration of this written form clause.</p><p align="justify" class="western">(3) Should any provision of this Agreement be or become legally invalid or if there is any void that needs to be filled, the validity of the remainder of the agreement shall not be affected thereby. Invalid provisions shall be replaced by common consent with such provisions which come as close as possible to the intended result of the invalid provision. In the event of gaps such provision shall come into force by common consent which comes as close as possible to the intended result of the agreement, should the matter have been considered in advance.</p><p align="justify">&nbsp;</p>',
		appendix:
			'<div class="pagebreak" style="break-before:always;"><p></p><h4 id="Orderprocessingagreement-Appendix1-TechnicalandOrganizationalMeasures">Appendix 1 - Technical and Organizational Measures&nbsp;</h4><p>System administrators are hereinafter referred to as "DevOps". The following Technical and Organizational Measures have been implemented:</p><ol><li>Entrance control: All systems are located in ISO 27001 certified&nbsp;data centers in Germany. Only DevOps are granted access to the physical systems.</li><li>Authentication access control: User access is secured with strong password protection according to the internal Password Policy or public key access control as well as second factor authentication (e.g. YubiKey).&nbsp;User access is managed by DevOps.</li><li>Authorization access control: Data records are secured with role based permissions. Permissions are managed by DevOps.</li><li>Data medium control: All hard discs containing personal data are encrypted. File permissions are allocated to DevOps users/roles as well as application users/roles to make sure no unauthorized access to files is allowed from logged in users and processes.</li><li>Transfer control: Transfer of personal data to other parties is being logged.&nbsp;Logs include the user/process that initiated the input, the type of personal data and the timestamp. The logs are kept for 6 months.</li><li>Input control: Input of new and updated as well as deletion of personal data is logged. Logs include the user/process that initiated the input, the type of personal data and the timestamp. The logs are kept for 6 months.</li><li>Transport control: Transport of personal data from and to the system are secured with strong SSL and/or end-to-end encryption.</li><li>Confidentiality: Personal data is stored end-to-end encrypted wherever possible.</li><li>Restoration control: All systems have a second network interface with access for DevOps only. This interface allows access even if the main interface is blocked. Components of the system can be restarted in case of error conditions. A DDOS mitigation service is automatically activated if a DDOS attack occurs that makes the system inaccessible.</li><li>Reliability:&nbsp;&nbsp;DevOps monitor all systems and are notified if any component of the system fails to be able to bring it up again immediately.</li><li>Data integrity: Automatic error correction on data mediums and also on database level make sure that data integrity is guaranteed. Additionally the integrity of end-to-end encrypted personal data is guaranteed through MACs during encryption and decryption.</li><li>Instruction control: All employees are aware of the purposes of processing and regularly complete&nbsp;an internal security awareness program. (Sub)processors are instructed by written contracts.</li><li>Availability control: All systems are located in ISO 27001 certified&nbsp;data centers in Germany which guarantee the physical availability and connection of the systems. All long-term data is stored as three replicas on different servers or in a RAID system. Backups are created prior to updating critical parts of the system.</li><li>Separability: Separate processing for personal data is set up as required.</li><li>Resilience: All systems use highly scalable components that are designed for much higher load than actually needed. All systems are expandable very quickly to continuously allow processing higher loads.</li></ol></div>\n' +
			"</div>",
	},
	"1_de": {
		heading:
			'<div class="papertext"><h2 style="text-align: center;" id="VertragzurAuftragsverarbeitung-VertragzurAuftragsverarbeitung">Vertrag zur Auftragsverarbeitung</h2><p style="text-align: center;">zwischen</p>',
		content:
			'<p style="text-align: center;">-&nbsp;Verantwortlicher -<br>nachstehend Auftraggeber genannt&nbsp;</p><p style="text-align: center;">und</p><p style="text-align: center;">Tutao GmbH, Deisterstr. 17a, 30449 Hannover</p><p style="text-align: center;">-&nbsp;Auftragsverarbeiter -<br>nachstehend&nbsp;Auftragnehmer genannt</p><p style="text-align: center;">&nbsp;</p><h2 id="VertragzurAuftragsverarbeitung-1.GegenstandundDauer">1.&nbsp;Gegenstand und Dauer</h2><p>Der Gegenstand des Auftrags ergibt sich aus den AGB der Tutao GmbH in der jeweils gültigen Version, siehe <span class="nolink">https://tuta.com/terms</span>, auf die hier verwiesen wird (im Folgenden Leistungsvereinbarung). Der&nbsp;Auftragnehmer verarbeitet dabei personenbezogene Daten für den Auftraggeber&nbsp;im Sinne von Art. 4 Nr. 2 und Art. 28 DS-GVO auf Grundlage dieses Vertrages.</p><p>Die Dauer dieses Auftrags entspricht der im jeweiligen Tarif gewählten Vertragslaufzeit.</p><h2 id="VertragzurAuftragsverarbeitung-2.Zweck,DatenkategorienundbetroffenePersonen">2. Zweck, Datenkategorien und betroffene Personen</h2><p>Zur Begründung eines Vertragsverhältnisses, und zur Leistungserbringung wird</p><ul><li>die neu registrierte E-Mail-Adresse</li></ul><p>als Bestandsdatum erfasst.</p><p>Für die Rechnungsstellung und Bestimmung der Umsatzsteuer&nbsp;werden</p><ul><li>der Sitz des Kunden (Land)</li><li>die Rechnungsadresse</li><li>die&nbsp;USt-IdNr. (nur für Geschäftskunden bestimmter Länder)</li></ul><p>als Bestandsdaten erfasst.</p><p>Zur Abwicklung von Zahlungen werden, je nach gewählter Zahlungsart, die folgenden Zahlungsdaten (Bestandsdaten) erfasst:</p><ul><li>Bankverbindung (Kontonummer und BLZ bzw. IBAN/BIC, ggf. Bankname, Kontoinhaber),</li><li>Kreditkartendaten,</li><li>der PayPal-Nutzername.</li></ul><p>Zur Abwicklung von Lastschriften wird die Bankverbindung an das beauftragte Kreditinstitut weitergegeben. <span>Zur Abwicklung von PayPal-Zahlungen werden die PayPal-Zahlungsdaten an PayPal (Europe) weitergegeben. </span>Zur Abwicklung von&nbsp;Kreditkartenzahlungen werden die Kreditkartendaten zur Auftragsverarbeitung an den Zahlungsdienstleister&nbsp;Braintree&nbsp;weitergegeben. Hierbei handelt es sich um eine Übermittlung von personenbezogenen Daten an ein Drittland. Ein mit Braintree geschlossener Vertrag sieht geeignete Garantien vor und stellt sicher, dass die weitergegebenen Daten nur im Einklang mit der DSGVO und lediglich zur Abwicklung von Zahlungen verwendet werden. Dieser Vertrag kann&nbsp;hier eingesehen werden:&nbsp;<span class="nolink">https://www.braintreepayments.com/assets/Braintree-PSA-Model-Clauses-March2018.pdf</span></p><p>Tutanota stellt Dienste zur Speicherung, Bearbeitung, Darstellung und elektronischem Versand von Daten bereit, wie z.B. E-Mail-Service, Kontaktverwaltung und Datenablage. Im Rahmen dieser Inhaltsdaten können personenbezogene Daten des Auftraggebers verarbeitet werden. Alle textuellen Inhalte werden verschlüsselt für den Nutzer und dessen Kommunikationspartner gespeichert, so dass die Tutao GmbH selbst keinen Zugriff auf diese Daten hat.</p><p>Zur Aufrechterhaltung des&nbsp;Mailserver-Betriebs, zur Fehlerdiagnose und zur Verhinderung von Missbrauch werden Mail-Server-Logs maximal 30 Tage gespeichert. Diese enthalten Sender- und Empfänger-E-Mail-Adressen sowie den Zeitpunkt der Verbindung, jedoch keine IP-Adressen der Benutzer.</p><p>Zur Sicherstellung des Betriebs, zur&nbsp;Verhinderung von Missbrauch und zur&nbsp;Besucherauswertung werden IP-Adressen der Benutzer verarbeitet. <span>Eine Speicherung erfolgt nur für anonymisierte und damit nicht mehr </span><span>personenbezogene </span><span>IP-Adressen.</span></p><p>Mit Ausnahme der Zahlungsdaten werden die personenbezogenen Daten inklusive der E-Mail-Adresse nicht an Dritte weitergegeben. Jedoch kann Tutao GmbH rechtlich verpflichtet werden Inhaltsdaten (bei Vorlage eines gültigen deutschen Gerichtsbeschlusses) sowie&nbsp;Bestandsdaten an Strafverfolgungsbehörden auszuliefern. Es erfolgt kein Verkauf von Daten.</p><p>Die Erbringung der vertraglich vereinbarten Datenverarbeitung findet ausschließlich in einem Mitgliedsstaat der Europäischen Union oder in einem anderen Vertragsstaat des Abkommens über den Europäischen Wirtschaftsraum statt.&nbsp;Jede Verlagerung in ein Drittland bedarf der vorherigen Zustimmung des Auftraggebers und darf nur erfolgen, wenn die besonderen Voraussetzungen der Art. 44 ff. DS-GVO erfüllt sind.&nbsp;</p><p>Die Kategorien der durch die Verarbeitung betroffenen Personen umfassen die durch den Auftraggeber in Tutanota eingerichtete Nutzer und deren Kommunikationspartner.</p><h2 id="VertragzurAuftragsverarbeitung-3.Technisch-organisatorischeMaßnahmen">3. Technisch-organisatorische Maßnahmen</h2><p>(1) Der&nbsp;Auftragnehmer hat die Umsetzung der im Vorfeld der Auftragsvergabe dargelegten und erforderlichen technischen und organisatorischen Maßnahmen vor Beginn der Verarbeitung, insbesondere hinsichtlich der konkreten Auftragsdurchführung zu dokumentieren und dem&nbsp;Auftraggeber zur Prüfung zu übergeben. Bei Akzeptanz durch den&nbsp;Auftraggeber&nbsp;werden die dokumentierten Maßnahmen Grundlage des Auftrags. Soweit die Prüfung des&nbsp;Auftraggebers einen Anpassungsbedarf ergibt, ist dieser einvernehmlich umzusetzen</p><p align="justify">(2) Der Auftragnehmer hat die Sicherheit gem. Art. 28 Abs. 3 lit. c, 32 DS-GVO insbesondere in Verbindung mit Art. 5 Abs. 1, Abs. 2 DS-GVO herzustellen. Insgesamt handelt es sich bei den zu treffenden Maßnahmen um Maßnahmen der Datensicherheit und zur Gewährleistung eines dem Risiko angemessenen Schutzniveaus hinsichtlich der Vertraulichkeit, der Integrität, der Verfügbarkeit sowie der Belastbarkeit der Systeme. Dabei sind der Stand der Technik, die Implementierungskosten und die Art, der Umfang und die Zwecke der Verarbeitung sowie die unterschiedliche Eintrittswahrscheinlichkeit und Schwere des Risikos für die Rechte und Freiheiten natürlicher Personen im Sinne von Art. 32 Abs. 1 DS-GVO zu berücksichtigen [Einzelheiten in Anlage 1].</p><p align="justify">(3) Die technischen und organisatorischen Maßnahmen unterliegen dem technischen Fortschritt und der Weiterentwicklung. Insoweit ist es dem Auftragnehmer gestattet, alternative adäquate Maßnahmen umzusetzen. Dabei darf das Sicherheitsniveau der festgelegten Maßnahmen nicht unterschritten werden. Wesentliche Änderungen sind zu dokumentieren.</p><h2 id="VertragzurAuftragsverarbeitung-4.Berichtigung,EinschränkungundLöschungvonDaten">4. Berichtigung, Einschränkung und Löschung von Daten</h2><p align="justify">(1) Der Auftragnehmer darf die Daten, die im Auftrag verarbeitet werden, nicht eigenmächtig sondern nur nach dokumentierter Weisung des Auftraggebers berichtigen, löschen oder deren Verarbeitung einschränken. Soweit eine betroffene Person sich diesbezüglich unmittelbar an den Auftragnehmer wendet, wird der Auftragnehmer dieses Ersuchen unverzüglich an den Auftraggeber weiterleiten.</p><p align="justify">(2) Soweit vom Leistungsumfang umfasst, sind Löschkonzept, Recht auf Vergessenwerden, Berichtigung, Datenportabilität und Auskunft nach dokumentierter Weisung des Auftraggebers unmittelbar durch den Auftragnehmer sicherzustellen.</p><h2 id="VertragzurAuftragsverarbeitung-5.QualitätssicherungundsonstigePflichtendesAuftragnehmers">5. Qualitätssicherung und sonstige Pflichten des Auftragnehmers</h2><p align="justify">Der Auftragnehmer hat zusätzlich zu der Einhaltung der Regelungen dieses Auftrags gesetzliche Pflichten gemäß Art. 28 bis 33 DS-GVO; insofern gewährleistet er insbesondere die Einhaltung folgender Vorgaben:</p><ol><li><p align="justify">Der Auftragnehmer ist nicht zur Bestellung eines Datenschutzbeauftragten verpflichtet. Als Ansprechpartner beim Auftragnehmer wird Herr Arne Möhle, Telefon: 0511 202801-11, arne.moehle@tutao.de, benannt.</p></li><li><p align="justify">Die Wahrung der Vertraulichkeit gemäß Art. 28 Abs. 3 S. 2 lit. b, 29, 32 Abs. 4 DS-GVO. Der Auftragnehmer setzt bei der Durchführung der Arbeiten nur Beschäftigte ein, die auf die Vertraulichkeit verpflichtet und zuvor mit den für sie relevanten Bestimmungen zum Datenschutz vertraut gemacht wurden. Der Auftragnehmer und jede dem Auftragnehmer unterstellte Person, die Zugang zu personenbezogenen Daten hat, dürfen diese Daten ausschließlich entsprechend der Weisung des Auftraggebers verarbeiten einschließlich der in diesem Vertrag eingeräumten Befugnisse, es sei denn, dass sie gesetzlich zur Verarbeitung verpflichtet sind.</p></li><li><p align="justify">Die Umsetzung und Einhaltung aller für diesen Auftrag erforderlichen technischen und organisatorischen Maßnahmen gemäß Art. 28 Abs. 3 S. 2 lit. c, 32 DS-GVO [Einzelheiten in Anlage 1].</p></li><li><p align="justify">Der Auftraggeber und der Auftragnehmer arbeiten auf Anfrage mit der Aufsichtsbehörde bei der Erfüllung ihrer Aufgaben zusammen.</p></li><li><p align="justify">Die unverzügliche Information des Auftragnehmers über Kontrollhandlungen und Maßnahmen der Aufsichtsbehörde, soweit sie sich auf diesen Auftrag beziehen. Dies gilt auch, soweit eine zuständige Behörde im Rahmen eines Ordnungswidrigkeits- oder Strafverfahrens in Bezug auf die Verarbeitung personenbezogener Daten bei der Auftragsverarbeitung beim Auftragnehmer ermittelt.</p></li><li><p align="justify">Soweit der Auftraggeber seinerseits einer Kontrolle der Aufsichtsbehörde, einem Ordnungswidrigkeits- oder Strafverfahren, dem Haftungsanspruch einer betroffenen Person oder eines Dritten oder einem anderen Anspruch im Zusammenhang mit der Auftragsverarbeitung beim Auftragnehmer ausgesetzt ist, hat ihn der Auftragnehmer nach besten Kräften zu unterstützen.</p></li><li><p align="justify">Der Auftragnehmer kontrolliert regelmäßig die internen Prozesse sowie die technischen und organisatorischen Maßnahmen, um zu gewährleisten, dass die Verarbeitung in seinem Verantwortungsbereich im Einklang mit den Anforderungen des geltenden Datenschutzrechts erfolgt und der Schutz der Rechte der betroffenen Person gewährleistet wird.</p></li><li><p align="justify">Nachweisbarkeit der getroffenen technischen und organisatorischen Maßnahmen gegenüber dem Auftraggeber im Rahmen seiner Kontrollbefugnisse nach Ziffer 7 dieses Vertrages.</p></li></ol><h2 id="VertragzurAuftragsverarbeitung-6.Unterauftragsverhältnisse">6. Unterauftragsverhältnisse</h2><p align="justify">(1) Als Unterauftragsverhältnisse im Sinne dieser Regelung sind solche Dienstleistungen zu verstehen, die sich unmittelbar auf die Erbringung der Hauptleistung beziehen. Nicht hierzu gehören Nebenleistungen, die der Auftragnehmer wie z.B. Telekommunikationsleistungen, Post-/Transportdienstleistungen, Wartung und Benutzerservice oder die Entsorgung von Datenträgern sowie sonstige Maßnahmen zur Sicherstellung der Vertraulichkeit, Verfügbarkeit, Integrität und Belastbarkeit der Hard- und Software von Datenverarbeitungsanlagen in Anspruch nimmt. Der Auftragnehmer ist jedoch verpflichtet, zur Gewährleistung des Datenschutzes und der Datensicherheit der Daten des Auftraggebers auch bei ausgelagerten Nebenleistungen angemessene und gesetzeskonforme vertragliche Vereinbarungen sowie Kontrollmaßnahmen zu ergreifen.</p><p align="justify">(2) Der Auftragnehmer darf Unterauftragnehmer (weitere Auftragsverarbeiter) nur nach vorheriger ausdrücklicher schriftlicher bzw. dokumentierter Zustimmung des Auftraggebers beauftragen.</p><p align="justify">(3) Die Auslagerung auf Unterauftragnehmer sowie der&nbsp;Wechsel der bestehenden Unterauftragnehmer sind zulässig, soweit:</p><ul><li>der Auftragnehmer eine solche Auslagerung auf Unterauftragnehmer dem Auftraggeber eine angemessene Zeit vorab schriftlich oder in Textform anzeigt und</li><li>der Auftraggeber nicht bis zum Zeitpunkt der Übergabe der Daten gegenüber dem Auftragnehmer schriftlich oder in Textform Einspruch gegen die geplante Auslagerung erhebt und</li><li>eine vertragliche Vereinbarung nach Maßgabe des Art. 28 Abs. 2-4 DS-GVO zugrunde gelegt wird.</li></ul><p align="justify">(4) Die Weitergabe von personenbezogenen Daten des Auftraggebers an den Unterauftragnehmer und dessen erstmaliges Tätigwerden sind erst mit Vorliegen aller Voraussetzungen für eine Unterbeauftragung gestattet.</p><p align="justify">(5) Erbringt der Unterauftragnehmer die vereinbarte Leistung außerhalb der EU/des EWR stellt der Auftragnehmer die datenschutzrechtliche Zulässigkeit durch entsprechende Maßnahmen sicher. Gleiches gilt, wenn Dienstleister im Sinne von Abs. 1 Satz 2 eingesetzt werden sollen.</p><p align="justify">(6) Eine weitere Auslagerung durch den Unterauftragnehmer bedarf der ausdrücklichen Zustimmung des Hauptauftraggebers (mind. Textform).</p><p align="justify">(7) Sämtliche vertraglichen Regelungen in der Vertragskette sind auch dem weiteren Unterauftragnehmer aufzuerlegen.</p><h2 class="western" id="VertragzurAuftragsverarbeitung-7.KontrollrechtedesAuftraggebers">7. Kontrollrechte des Auftraggebers</h2><p align="justify">(1) Der Auftraggeber hat das Recht, im Benehmen mit dem Auftragnehmer Überprüfungen durchzuführen oder durch im Einzelfall zu benennende Prüfer durchführen zu lassen. Er hat das Recht, sich durch Stichprobenkontrollen, die in der Regel rechtzeitig anzumelden sind, von der Einhaltung dieser Vereinbarung durch den Auftragnehmer in dessen Geschäftsbetrieb zu überzeugen.</p><p align="justify">(2) Der Auftragnehmer stellt sicher, dass sich der Auftraggeber von der Einhaltung der Pflichten des Auftragnehmers nach Art. 28 DS-GVO überzeugen kann. Der Auftragnehmer verpflichtet sich, dem Auftraggeber auf Anforderung die erforderlichen Auskünfte zu erteilen und insbesondere die Umsetzung der technischen und organisatorischen Maßnahmen nachzuweisen.</p><p align="justify">(3) Der Nachweis solcher Maßnahmen, die nicht nur den konkreten Auftrag betreffen, kann erfolgen durch</p><ul><li>die Einhaltung genehmigter Verhaltensregeln gemäß Art. 40 DS-GVO;</li><li>die Zertifizierung nach einem genehmigten Zertifizierungsverfahren gemäß Art. 42 DS-GVO;</li><li>aktuelle Testate, Berichte oder Berichtsauszüge unabhängiger Instanzen (z.B. Wirtschaftsprüfer, Revision, Datenschutzbeauftragter, IT-Sicherheitsabteilung, Datenschutzauditoren, Qualitätsauditoren);</li><li>eine geeignete Zertifizierung durch IT-Sicherheits- oder Datenschutzaudit (z.B. nach BSI-Grundschutz).</li></ul><p align="justify">(4) Für die Ermöglichung von Kontrollen durch den Auftraggeber kann der Auftragnehmer einen Vergütungsanspruch geltend machen.</p><h2 class="western" id="VertragzurAuftragsverarbeitung-8.MitteilungbeiVerstößendesAuftragnehmers">8. Mitteilung bei Verstößen des Auftragnehmers</h2><p align="justify">(1) Der Auftragnehmer unterstützt den Auftraggeber bei der Einhaltung der in den Artikeln 32 bis 36 der DS-GVO genannten Pflichten zur Sicherheit personenbezogener Daten, Meldepflichten bei Datenpannen, Datenschutz-Folgeabschätzungen und vorherige Konsultationen. Hierzu gehören u.a.</p><ol><li><p align="justify">die Sicherstellung eines angemessenen Schutzniveaus durch technische und organisatorische Maßnahmen, die die Umstände und Zwecke der Verarbeitung sowie die prognostizierte Wahrscheinlichkeit und Schwere einer möglichen Rechtsverletzung durch Sicherheitslücken berücksichtigen und eine sofortige Feststellung von relevanten Verletzungsereignissen ermöglichen</p></li><li><p align="justify">die Verpflichtung, Verletzungen personenbezogener Daten unverzüglich an den Auftraggeber zu melden</p></li><li><p align="justify">die Verpflichtung, dem Auftraggeber im Rahmen seiner Informationspflicht gegenüber dem Betroffenen zu unterstützen und ihm in diesem Zusammenhang sämtliche relevante Informationen unverzüglich zur Verfügung zu stellen</p></li><li><p align="justify">die Unterstützung des Auftraggebers für dessen Datenschutz-Folgenabschätzung</p></li><li><p align="justify">die Unterstützung des Auftraggebers im Rahmen vorheriger Konsultationen mit der Aufsichtsbehörde</p></li></ol><p align="justify">(2) Für Unterstützungsleistungen, die nicht in der Leistungsbeschreibung enthalten oder nicht auf ein Fehlverhalten des Auftragnehmers zurückzuführen sind, kann der Auftragnehmer eine Vergütung beanspruchen.</p><h2 class="western" id="VertragzurAuftragsverarbeitung-9.WeisungsbefugnisdesAuftraggebers">9. Weisungsbefugnis des Auftraggebers</h2><p align="justify">(1) Mündliche Weisungen bestätigt der Auftraggeber unverzüglich (mind. Textform).</p><p align="justify">(2) Der Auftragnehmer hat den Auftraggeber unverzüglich zu informieren, wenn er der Meinung ist, eine Weisung verstoße gegen Datenschutzvorschriften. Der Auftragnehmer ist berechtigt, die Durchführung der entsprechenden Weisung solange auszusetzen, bis sie durch den Auftraggeber bestätigt oder geändert wird.</p><h2 class="western" id="VertragzurAuftragsverarbeitung-10.LöschungundRückgabevonpersonenbezogenenDaten">10. Löschung und Rückgabe von personenbezogenen Daten</h2><p align="justify">(1) Kopien oder Duplikate der Daten werden ohne Wissen des Auftraggebers nicht erstellt. Hiervon ausgenommen sind Sicherheitskopien, soweit sie zur Gewährleistung einer ordnungsgemäßen Datenverarbeitung erforderlich sind, sowie Daten, die im Hinblick auf die Einhaltung gesetzlicher Aufbewahrungspflichten erforderlich sind.</p><p align="justify">(2) Nach Abschluss der vertraglich vereinbarten Arbeiten oder früher nach Aufforderung durch den Auftraggeber – spätestens mit Beendigung der Leistungsvereinbarung – hat der Auftragnehmer sämtliche in seinen Besitz gelangten Unterlagen, erstellte Verarbeitungs- und Nutzungsergebnisse sowie Datenbestände, die im Zusammenhang mit dem Auftragsverhältnis stehen, dem Auftraggeber auszuhändigen oder nach vorheriger Zustimmung datenschutzgerecht zu vernichten. Gleiches gilt für Test- und Ausschussmaterial. Das Protokoll der Löschung ist auf Anforderung vorzulegen.</p><p align="justify">(3) Dokumentationen, die dem Nachweis der auftrags- und ordnungsgemäßen Datenverarbeitung dienen, sind durch den Auftragnehmer entsprechend der jeweiligen Aufbewahrungsfristen über das Vertragsende hinaus aufzubewahren. Er kann sie zu seiner Entlastung bei Vertragsende dem Auftraggeber übergeben.</p><h2 id="VertragzurAuftragsverarbeitung-11.Schlussbestimmungen">11. Schlussbestimmungen</h2><p align="justify">(1) <span>Dieser Vertrag unterliegt dem Recht der Bundesrepublik Deutschland. Gerichtsstand ist Hannover.</span></p><p align="justify"><span>(2) Änderungen und Ergänzungen dieses Vertrags bedürfen der Schriftform. Dies gilt auch für den Verzicht auf das Schriftformerfordernis.</span></p><p align="justify" class="western">(3) <span>Sollten einzelne Bestimmungen dieses Vertrags unwirksam sein oder werden, so wird dadurch die Gültigkeit der übrigen Bestimmungen nicht berührt. Die Vertragsparteien verpflichten sich in diesen Fällen, anstelle der etwa unwirksamen Bestimmung(en) – mit Wirkung von Beginn der Unwirksamkeit an – eine Ersatzregelung oder ggf. einen neuen wirksamen Vertrag zu vereinbaren, die bzw. der dem wirtschaftlichen gewollten Zweck der unwirksamen Bestimmung(en) weitgehend entspricht oder am nächsten kommt. Dies gilt auch für den Fall, dass der Vertrag eine Regelungslücke enthalten sollte.</span></p><p align="justify">&nbsp;</p>',
		appendix:
			'<div class="pagebreak" style="break-before:always;"><p></p><h2 id="VertragzurAuftragsverarbeitung-Anlage1(TechnischeundorganisatorischeMaßnahmen)">Anlage 1 (Technische und organisatorische Maßnahmen)</h2><p>Die Systemadministratoren werden im Folgenden "DevOps" genannt. Folgende Maßnahmen wurden getroffen:</p><ol><li>Zutrittskontrolle: Alle Systeme sind in ISO 27001 zertifizierten Rechenzentren in Deutschland gehostet. Nur DevOps haben Zutritt zu den physischen Systemen.</li><li>Zugangskontrolle/Benutzerkontrolle: Der Zugriff durch Benutzer ist mit starken Passwörtern entsprechend den internen Passwortregeln oder Public-Key-Zugriff und Zwei-Faktor-Authentifizierung (e.g. YubiKey) gesichert.&nbsp;Benutzerzugriff wird von DevOps verwaltet.</li><li>Zugriffskontrolle/Speicherkontrolle: Datensätze sind mit rollenbasierten Berechtigungen geschützt. Berechtigungen werden von DevOps verwaltet.</li><li>Datenträgerkontrolle: <span>Alle Festplatten mit personenbezogenen Daten sind verschüsselt. Dateiberechtigungen sind für DevOps sowie Anwendungen so vergeben, dass unberechtigter Zugriff auf Dateien von eingeloggten Benutzern und Prozessen verhindert wird.</span></li><li>Übertragungskontrolle/Weitergabekontrolle: Weitergabe von personenbezogenen Daten an andere Empfänger wird protokolliert.&nbsp;Die Protokolle enthalten den Benutzer/Prozess, der die Eingabe initiiert hat, die Kategorie personenbezogener Daten und den Zeitstempel. Die Protokolle werden für sechs Monate aufgehoben.</li><li>Eingabekontrolle: Eingabe von neuen und aktualisierten sowie die Löschung von personenbezogenen Daten wird protokolliert. <span>Die Protokolle enthalten den Benutzer/Prozess, der die Eingabe initiiert hat, die Kategorie personenbezogener Daten und den Zeitstempel. Die Protokolle werden für sechs Monate aufgehoben.</span></li><li>Transportkontrolle: Übertragung von personenbezogenen Daten von und zu den Systemen ist durch starke SSL-Verschlüsselung und/oder Ende-zu-Ende-Verschlüsselung gesichert.</li><li>Vertraulichkeit: Personenbezogene Daten werden soweit möglich Ende-zu-Ende verschlüsselte gespeichert.</li><li>Wiederherstellbarkeit: Alle Systeme haben eine zweite Netzwerkschnittstelle, die nur den Zugriff von DevOps erlaubt. Diese Schnittstelle erlaubt den Zugriff selbst wenn die Hauptschnittstelle blockiert ist. Komponenten des Systems können im Fehlerfall neu gestartet werden. Ein Dienst zum Schutz vor DDOS-Angriffen wird automatisch gestartet, wenn solch ein Angriff erkannt wird.</li><li>Zuverlässigkeit:&nbsp;&nbsp;DevOps überwachen alle Systeme und werden automatisch benachrichtigt, wenn eine Komponente des Systems ausfällt, um diese sofort wieder aktivieren zu können.</li><li>Datenintegrität: Automatische Fehlerkorrektur auf Datenträgern und auf Datenbankebene stellt sicher, dass die Datenintegrität gewahrt bleibt. Zusätzlich wird die Integrität der Ende-zu-Ende verschlüsselten personenbezogenen Daten durch MACs bei der Ver- und Entschlüsselung sichergestellt.</li><li>Auftragskontrolle: Alle Mitarbeiter kennen die Zwecke der Verarbeitung und absolvieren regelmäßig ein internes Sicherheitstraining. Unterauftragnehmer werden nur schriftlich beauftragt.</li><li>Verfügbarkeitskontrolle: <span>Alle Systeme sind in ISO 27001 zertifizierten Rechenzentren in Deutschland gehostet, die die physische Verfügbarkeit und Verbindung der Systeme sicherstellen</span>. Alle langfristig gespeicherten Daten werden dreifach repliziert auf unterschiedlichen Servern oder in einem RAID-System abgelegt. Vor der Aktualisierung kritischer Teile des Systems werden Backups angelegt.</li><li>Trennbarkeit: Getrennte Verarbeitung von personenbezogenen Daten ist bedarfsabhängig eingerichtet.</li><li>Belastbarkeit: Alle Systeme bestehen aus hochskalierbaren Komponenten, die für viel höhere Lasten als tatsächlich benötigt ausgelegt sind. Alle Systeme sind kurzfristig erweiterbar, um kontinuierlich steigende Lasten verarbeiten zu können.</li></ol></div>\n' +
			"</div>",
	},
}

export function showForSigning(customer: Customer, accountingInfo: AccountingInfo) {
	const signAction = (dialog: Dialog) => {
		let data = createSignOrderProcessingAgreementData({
			version: version,
			customerAddress: addressEditor.getValue(),
		})

		if (addressEditor.getValue().trim().split("\n").length < 3) {
			Dialog.message("contractorInfo_msg")
		} else {
			locator.serviceExecutor.post(SignOrderProcessingAgreementService, data).then(() => dialog.close())
		}
	}

	const version = "1_" + (lang.code === "de" ? "de" : "en")
	const addressEditor = new HtmlEditor()
		.setMinHeight(120)
		.showBorders()
		.setPlaceholderId("contractor_label")
		.setMode(HtmlEditorMode.HTML)
		.setHtmlMonospace(false)
		.setValue(formatNameAndAddress(accountingInfo.invoiceName, accountingInfo.invoiceAddress))
	Dialog.showActionDialog({
		title: lang.get("orderProcessingAgreement_label"),
		okAction: signAction,
		okActionTextId: "sign_action",
		type: DialogType.EditLarge,
		child: () => {
			// @ts-ignore
			const text = agreementTexts[version]
			return m(".pt", [
				m.trust(text.heading),
				m(".flex-center", m(".dialog-width-s", [m(addressEditor), m(".small", lang.get("contractorInfo_msg"))])),
				m.trust(text.content),
				m.trust(text.appendix),
			])
		},
	})
}

// this is necessary because some style combinations
// cause several browsers not to print
// the content below the fold
function printElementContent(elem: HTMLElement | null) {
	const root = document.getElementById("root")
	const body = document.body
	if (!elem || !root || !body) return
	let printDiv = document.getElementById(PRINT_DIV_ID)

	if (!printDiv) {
		printDiv = document.createElement("DIV")
		printDiv.id = PRINT_DIV_ID
		body.appendChild(printDiv)
		const classes = root.className.split(" ")
		classes.push("noprint")
		root.className = classes.join(" ")
	}

	printDiv.innerHTML = elem.innerHTML
	printDiv.classList.add("noscreen")
	window.print()
}

function cleanupPrintElement() {
	const root = document.getElementById("root")
	const body = document.body
	const printDiv = document.getElementById(PRINT_DIV_ID)
	if (!printDiv || !root || !body) return
	body.removeChild(printDiv)
	root.className = root.className
		.split(" ")
		.filter((c) => c !== "noprint")
		.join(" ")
}

export function showForViewing(agreement: OrderProcessingAgreement, signerUserGroupInfo: GroupInfo) {
	Dialog.showActionDialog({
		title: lang.get("orderProcessingAgreement_label"),
		okAction: !isApp() && "function" === typeof window.print ? () => printElementContent(document.getElementById("agreement-content")) : null,
		okActionTextId: "print_action",
		cancelActionTextId: "close_alt",
		type: DialogType.EditLarge,
		child: () => {
			// @ts-ignore
			const text = agreementTexts[agreement.version]
			return m(
				"#agreement-content.pt",
				{
					onremove: cleanupPrintElement,
				},
				[
					m.trust(text.heading),
					m("p.text-center.text-prewrap", agreement.customerAddress),
					m.trust(text.content),
					m(
						"i",
						lang.get("signedOn_msg", {
							"{date}": formatDate(agreement.signatureDate),
						}) +
							" " +
							lang.get("by_label") +
							" " +
							getMailAddressDisplayText(signerUserGroupInfo.name, neverNull(signerUserGroupInfo.mailAddress), false),
					),
					m("hr"),
					m.trust(text.appendix),
				],
			)
		},
	})
}
